#!/bin/bash

# Browsershot Installation Script
# This script installs Spatie Browsershot and its dependencies

set -e  # Exit on any error

echo "======================================"
echo "Browsershot PDF Generator Installation"
echo "======================================"
echo ""

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Check if we're in the correct directory
if [ ! -f "artisan" ]; then
    echo -e "${RED}Error: artisan file not found. Please run this script from the Laravel root directory.${NC}"
    exit 1
fi

echo -e "${GREEN}Step 1: Installing Browsershot via Composer${NC}"
echo "Running: composer require spatie/browsershot"
composer require spatie/browsershot

if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to install Browsershot via Composer${NC}"
    exit 1
fi

echo ""
echo -e "${GREEN}Step 2: Checking Node.js installation${NC}"

# Check if Node.js is installed
if ! command -v node &> /dev/null; then
    echo -e "${YELLOW}Node.js is not installed.${NC}"
    echo "Please install Node.js from: https://nodejs.org/"
    echo "After installing Node.js, run: npm install puppeteer"
    exit 1
else
    NODE_VERSION=$(node -v)
    echo "Node.js is installed: $NODE_VERSION"
fi

# Check if npm is installed
if ! command -v npm &> /dev/null; then
    echo -e "${RED}npm is not installed.${NC}"
    exit 1
else
    NPM_VERSION=$(npm -v)
    echo "npm is installed: $NPM_VERSION"
fi

echo ""
echo -e "${GREEN}Step 3: Installing Puppeteer${NC}"
echo "This may take a few minutes as it downloads Chromium..."

# Check if package.json exists, create if not
if [ ! -f "package.json" ]; then
    echo "Creating package.json..."
    npm init -y
fi

# Install Puppeteer
npm install puppeteer

if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to install Puppeteer${NC}"
    exit 1
fi

echo ""
echo -e "${GREEN}Step 4: Testing Browsershot Installation${NC}"

# Create a test PHP script
cat > test_browsershot.php <<'EOF'
<?php
require __DIR__.'/vendor/autoload.php';

use Spatie\Browsershot\Browsershot;

try {
    echo "Testing Browsershot...\n";

    $pdf = Browsershot::html('<html><body><h1>Test PDF</h1><p>If you can read this, Browsershot is working!</p></body></html>')
        ->pdf();

    file_put_contents('test_browsershot_output.pdf', $pdf);

    if (file_exists('test_browsershot_output.pdf')) {
        echo "✅ Success! Test PDF generated: test_browsershot_output.pdf\n";
        echo "File size: " . filesize('test_browsershot_output.pdf') . " bytes\n";
        exit(0);
    } else {
        echo "❌ Error: PDF file was not created\n";
        exit(1);
    }

} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}
EOF

# Run the test
php test_browsershot.php

if [ $? -eq 0 ]; then
    echo ""
    echo -e "${GREEN}✅ Installation completed successfully!${NC}"
    echo ""
    echo "Next steps:"
    echo "1. Check the test PDF: test_browsershot_output.pdf"
    echo "2. Read the documentation: BROWSERSHOT_SETUP.md"
    echo "3. Test quote generation with a sample project"
    echo ""
    echo "Clean up test files with:"
    echo "  rm test_browsershot.php test_browsershot_output.pdf"

    # Clean up test script
    rm -f test_browsershot.php
else
    echo ""
    echo -e "${RED}Installation completed but test failed.${NC}"
    echo "Please check the error messages above."
    echo ""
    echo "Common issues:"
    echo "1. Chrome/Chromium not found - Install Puppeteer: npm install puppeteer"
    echo "2. Permission errors - Check file permissions"
    echo "3. Timeout errors - Increase timeout in BrowsershotPdfGenerator.php"

    rm -f test_browsershot.php
    exit 1
fi

echo ""
echo "======================================"
echo "Installation Complete"
echo "======================================"
