<?php
namespace MajorMedia\Project\Controllers;

use Backend;
use BackendMenu;
use Backend\Classes\Controller;
use MajorMedia\Project\Models\Project;
use MajorMedia\Project\Models\FrontDesign;
use MajorMedia\Project\Classes\QuoteGenerator;
use Flash;

class Projects extends Controller
{
    public $implement = [
        \Backend\Behaviors\FormController::class,
        \Backend\Behaviors\ListController::class,
        \Backend\Behaviors\RelationController::class,
    ];

    public $formConfig = 'config_form.yaml';
    public $listConfig = 'config_list.yaml';
    public $relationCofig = 'config_relation.yaml';

    public function __construct()
    {
        parent::__construct();
        BackendMenu::setContext('MajorMedia.Project', 'project', 'projects');

    }

    /**
     * AJAX handler to load the generate quote form popup
     */
    public function onLoadGenerateQuoteForm()
    {
        $projectId = post('project_id');
        $project = Project::with('user')->findOrFail($projectId);
        $frontDesigns = FrontDesign::where('is_active', 1)->get();

        $this->vars['projectId'] = $projectId;
        $this->vars['project'] = $project;
        $this->vars['frontDesigns'] = $frontDesigns;

        return [
            'result' => $this->makePartial('$/majormedia/project/controllers/projects/_generate_quote_popup.htm')
        ];
    }

    /**
     * AJAX handler to generate quote
     */
    public function onGenerateQuote()
    {
        try {
            $projectId = post('project_id');
            $clientName = post('client_name');
            $quoteDate = post('quote_date');
            $frontDesignId = post('front_design_id');
            $notes = post('notes');

            // Validate inputs
            if (!$projectId || !$clientName || !$quoteDate || !$frontDesignId) {
                throw new \ValidationException(['error' => 'Tous les champs obligatoires doivent être remplis.']);
            }

            // Get project
            $project = Project::findOrFail($projectId);

            // Generate quote using QuoteGenerator
            $generator = new QuoteGenerator();
            $quote = $generator->generateQuote($project, $frontDesignId);

            // Update quote with form data
            if ($quote) {
                $quote->client_name = $clientName;
                $quote->quote_date = $quoteDate;
                if ($notes) {
                    $quote->notes = $notes;
                }
                $quote->save();

                Flash::success("Devis {$quote->reference} généré avec succès!");

                return [
                    'success' => true,
                    'message' => "Devis {$quote->reference} généré avec succès!",
                    'quote_id' => $quote->id,
                    'reference' => $quote->reference
                ];
            }

            throw new \Exception('Erreur lors de la génération du devis.');

        } catch (\ValidationException $e) {
            Flash::error($e->getMessage());
            throw $e;
        } catch (\Exception $e) {
            \Log::error('Error generating quote: ' . $e->getMessage());
            Flash::error('Erreur lors de la génération du devis: ' . $e->getMessage());
            throw $e;
        }
    }

}
