<?php
namespace MajorMedia\Structure;

use Event;
use JwtAuth;
use Backend\Facades\Backend;
use System\Classes\PluginBase;
use Backend\Classes\Controller;
use MajorMedia\Structure\Models\Page;
use MajorMedia\Project\Models\Project;
use Illuminate\Support\Facades\Session;
use MajorMedia\Structure\Models\Feature;
use MajorMedia\Structure\Models\SiteMap;
use MajorMedia\Project\Models\ProjectType;
use MajorMedia\Project\Controllers\Projects;
use MajorMedia\Project\Controllers\ProjectTypes;
/**
 * Plugin class
 */
class Plugin extends PluginBase
{
    public $require = ['MajorMedia.Project'];
    /**
     * register method, called when the plugin is first registered.
     */
    public function register()
    {
    }

    /**
     * boot method, called right before the request route.
     */
    public function boot()
    {
        $this->extendProjectModel();
        $this->extendProjectTypeModel();
        $this->extendProjectTypeFormFields();
        $this->extendProjectController();
        $this->extendProjectFormFields();
        Event::listen('majormedia.project::extendSiteMapHidden', function () {
            Page::extend(function ($model) {
                $model->makeHidden(['pivot']);
            });
        });
        Event::listen('backend.menu.extendItems', function ($manager) {
            $manager->addSideMenuItems('MajorMedia.Projet', 'project', [
                'sitemaps' => [
                    'label' => 'Site Map',
                    'icon' => 'icon-copy',
                    'url' => Backend::url('majormedia/structure/sitemaps'),
                    'permissions' => ['majormedia.structure.structure'],
                ],
            ]);
        });

    }

    /**
     * registerComponents used by the frontend.
     */
    public function registerComponents()
    {
    }

    /**
     * registerSettings used by the backend.
     */
    public function registerSettings()
    {
    }

    public function extendProjectModel()
    {
        Project::extend(function ($model) {
            $model->append(['is_content_completed', 'structure', 'features']);
            $model->makeVisible(['pivot', 'structure', 'features', 'is_content_completed']);

            //accessors
            $model->addDynamicMethod('getStructureAttribute', function () use ($model) {
                $type = $model->project_type->id;

                $structure = $model->sitemap_pages()->with([
                    'sections' => function ($query) use ($type) {
                        $query->where('project_type_id', $type);
                    },
                ])->get();

                $formattedStructure = $structure->mapWithKeys(function ($page) {
                    $sectionIds = $page->sections->pluck('id')->toArray();
                    return [$page->id => $sectionIds];
                });
                // return null;
                return $formattedStructure->isNotEmpty() ? $formattedStructure : null;
            });
            $model->addDynamicMethod('getIsContentCompletedAttribute', function () {
                return 0;
            });

            $model->addDynamicMethod('getFeaturesAttribute', function () use ($model) {
                if (JWTAuth::getToken()) {  // Ensure a token is available
                    Session::put('required_token', true);
                }
                $projectId = $model->id;

                //get the linked features
                $featureSynced = $model->features()
                    ->wherePivot('project_id', $projectId)
                    ->withPivot(['is_required', 'is_primary'])
                    ->get();
                //get the customeCreatedFeatures
                $customFeatures = $model->custom_features()->get();
                return [
                    'features' => $featureSynced,
                    'customFeatures' => $customFeatures
                ];
            });

            //relations
            $model->belongsToMany['sitemap_pages'] = [
                Page::class,
                'table' => 'majormedia_structure_sitemaps',
                'key' => 'project_id',
                'otherKey' => 'page_id',
                'pivot' => ['content']
            ];

            $model->hasMany['sitemaps'] = [
                SiteMap::class
            ];

            $model->belongsToMany['features'] = [
                Feature::class,
                'table' => 'majormedia_structure_project_features',
                'key' => 'project_id',
                'otherKey' => 'feature_id',
                'pivot' => ['is_required', 'is_primary']
            ];
            $model->hasMany['custom_features'] = [Feature::class, 'key' => 'project_id'];

        });
    }

    public function extendProjectTypeModel()
    {
        ProjectType::extend(function ($model) {
            $model->belongsToMany['pages'] = [
                Page::class,
                'table' => 'majormedia_structure_project_pages',
                'key' => 'project_type_id',
                'otherKey' => 'page_id',
                'pivot' => ['is_required', 'is_primary', 'is_default']
            ];
            $model->belongsToMany['features'] = [
                Feature::class,
                'table' => 'majormedia_structure_project_types_features',
                'key' => 'project_type_id',
                'otherKey' => 'feature_id',
                'pivot' => ['is_required', 'is_primary']
            ];
        });
    }

    /**
     * extend Form Fields
     */
    public function extendProjectFormFields()
    {
        Event::listen('backend.form.extendFields', function ($widget) {
            // Only for the Project controller
            if (!$widget->getController() instanceof Projects) {
                return;
            }
            // Only for the Project model
            if (!$widget->model instanceof Project) {
                return;
            }

            // Add an extra Project Fields
            $widget->addTabFields([
                'custom_features' => [
                    'label' => 'Custom Features',
                    'type' => 'partial',
                    'path' => '$/majormedia/structure/partials/_custom_features.htm',
                    'tab' => 'Custom Features'
                ],
                'features' => [
                    'label' => 'Synced Features',
                    'type' => 'partial',
                    'path' => '$/majormedia/structure/partials/_features.htm',
                    'tab' => 'Synced Features'
                ],
                'sitemaps' => [
                    'label' => 'Site Maps',
                    'type' => 'partial',
                    'path' => '$/majormedia/structure/partials/_site_maps.htm',
                    'tab' => 'Site Maps'
                ],
                'sitemap_schema' => [
                    'label' => 'Sitemap Schema',
                    'type' => 'partial',
                    'path' => '$/majormedia/structure/partials/_sitemap_schema.htm',
                    'tab' => 'Sitemap Schema'
                ],
            ]);
        });
    }
    public function extendProjectTypeFormFields()
    {
        Event::listen('backend.form.extendFields', function ($widget) {
            // Only for the ProjectTypes controller
            if (!$widget->getController() instanceof ProjectTypes) {
                return;
            }
            // Only for the ProjectType model
            if (!$widget->model instanceof ProjectType) {
                return;
            }
            // Add an extra ProjectTypes column
            $widget->addTabFields([
                'pages' => [
                    'label' => 'Pages',
                    'type' => 'partial',
                    'path' => '$/majormedia/structure/partials/_pages.htm',
                    'tab' => 'Pages'
                ]
            ]);
        });
    }

    /**
     * exeten the Relation Config
     */
    protected function extendProjectController()
    {
        Projects::extend(function (Controller $controller) {
            // Config_list: Check if it's already set, else add it
            if (!isset($controller->listConfig)) {
                $controller->implement[] = 'Backend.Behaviors.ListController';
                $controller->addDynamicProperty('listConfig');
            }
            //TODO: find out why this causing a problem

            // if (!$controller->isClassExtendedWith('Backend.Behaviors.RelationController')) {
            //     $controller->implement[] = 'Backend.Behaviors.RelationController';
            // }

            if (!isset($controller->relationConfig)) {
                $controller->addDynamicProperty('relationConfig');
            }
            $controller->relationConfig = '$/majormedia/project/controllers/projects/config_relation.yaml';

            $relationConfigPath = '$/majormedia/structure/config/project_config_relation.yaml';

            $controller->relationConfig = $controller->mergeConfig(
                $controller->relationConfig,
                $relationConfigPath
            );

        });
    }

}
