<?php
namespace Majormedia\Structure\Http;

ini_set("memory_limit", "-1");
use Backend\Classes\Controller;
use MajorMedia\Project\Models\Project;
use MajorMedia\Structure\Models\Section;
use MajorMedia\Structure\Models\SiteMap;
use MajorMedia\ToolBox\Traits\JsonAbort;
use MajorMedia\ToolBox\Utility\ErrorCodes;
use MajorMedia\ToolBox\Traits\RetrieveUser;
use MajorMedia\ToolBox\Traits\GetValidatedInput;
use MajorMedia\ToolBox\Traits\GetFileFromBase64;
use Illuminate\Support\Facades\Session;
use Event;
/**
 * Site Maps Back-end Controller
 */
class SiteMaps extends Controller
{

    use RetrieveUser, GetValidatedInput, JsonAbort, GetFileFromBase64;
    public $implement = [
        'MajorMedia.ToolBox.Behaviors.RestController'
    ];

    public $restConfig = 'config_rest.yaml';

    public function index()
    {
        Session::put('required_token', true);
        $this->retrieveUser();
        $user = $this->user;

        $project = $user->project;
        $projectID = $project->id;
        if (!$project) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => ErrorCodes::PROJECT_NOT_FOUND,
            ], 404);
        }

        $projectType = $project->project_type;

        if (!$projectType) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => ErrorCodes::PROJECT_TYPE_NOT_VALID,
            ], 404);
        }

        $typeId = $projectType->id;

        Event::fire('majormedia.project::extendSiteMapHidden');
        \DB::statement('set session sql_mode=""');
        $structure = $project->sitemap_pages()
            ->with([
                'sitemap_sections' => function ($query) use ($projectID, $typeId) {

                    $query->wherePivot('project_id', $projectID)
                        ->addSelect('majormedia_structure_sections.*')
                        ->addSelect('majormedia_structure_sitemaps.page_id as sitemap_page_id')
                        ->selectSub(function ($subQuery) use ($typeId) {
                            $subQuery->select('is_repeater')
                                ->from('majormedia_structure_page_sections')
                                ->whereColumn('section_id', 'majormedia_structure_sections.id')
                                ->whereColumn('page_id', 'majormedia_structure_sitemaps.page_id')
                                ->where('project_type_id', $typeId)
                                ->limit(1);
                        }, 'is_repeater')
                        ->groupByRaw(
                            'majormedia_structure_sitemaps.project_id,
                    majormedia_structure_sitemaps.page_id,
                    majormedia_structure_sitemaps.section_id'
                        );
                },
                'category'
            ])
            ->distinct()
            ->get()
            ->unique('id');

        // Load sitemap_fields for each section filtered by its specific page_id
        $structure->each(function ($page) use ($projectID) {
            $page->sitemap_sections->each(function ($section) use ($projectID) {
                $pageId = $section->sitemap_page_id ?? $section->pivot->page_id;
                $section->setRelation('sitemap_fields',
                    $section->sitemap_fields()
                        ->wherePivot('project_id', $projectID)
                        ->wherePivot('page_id', $pageId)
                        ->orderBy('order', 'asc')
                        ->get()
                );
            });
        });

        $structure = $structure->groupBy(function ($page) {
            return $page->category ? $page->category->name : null;
        });

        return $this->JsonAbort([
            'status' => 'success',
            'structure' => $structure,
        ], 200);
    }

    public function store()
    {
        $this->retrieveUser();
        $user = $this->user;

        $data = $this->getValidatedInput(
            ['project_id', 'structure'],
            [
                'project_id' => 'nullable|integer',
                'structure' => 'nullable|array',
                'structure.*' => 'nullable|array',
                'structure.*.*' => 'nullable|integer',
            ]
        );
        $project = $user->project;

        $project = Project::find($project->id);
        if (!$project) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => ErrorCodes::PROJECT_NOT_FOUND
            ], 200);
        }

        $project->sitemap_pages()->detach();

        foreach ($data['structure'] as $pageId => $sectionIds) {


            if (empty($sectionIds)) {
                SiteMap::create([
                    'project_id' => $project->id,
                    'page_id' => $pageId,
                    'section_id' => null,
                    'field_id' => null,
                    'content' => null,
                ]);
                continue;
            }
            foreach ($sectionIds as $sectionId) {

                $sections = Section::find($sectionId);

                if ($sections) {
                    foreach ($sections->fields as $field) {
                        SiteMap::create([
                            'project_id' => $project->id,
                            'page_id' => $pageId,
                            'section_id' => $sectionId,
                            'field_id' => $field->id,
                            'content' => $field->content ?? null,
                        ]);
                    }
                }
            }
        }

        return $this->JsonAbort([
            'status' => 'success',
        ], 200);
    }


    public function getStructureProject()
    {
        $this->retrieveUser();
        $user = $this->user;
        $project = $user->project;
        $projectID = $project->id;

        $type = $project->project_type->id;

        $structure = $project->project_type->pages()->with([
            'sections' => function ($query) use ($type) {
                $query->where('project_type_id', $type);
            },
        ])->get();
        return $this->JsonAbort([
            'status' => 'success',
            'structure' => $structure,
        ], 201);
    }
    public function updateSiteMap()
    {
        $this->retrieveUser();
        $user = $this->user;
        $project = $user->project;

        $data = $this->getValidatedInput(
            ['structure'],
            [
                'structure' => 'nullable|array',
                'structure.*' => 'nullable|array',
                'structure.*.*' => 'nullable|array',
                'structure.*.*.*' => 'nullable|string',
            ]
        );
        $project = Project::find($project->id);

        if (!$project) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => ErrorCodes::PROJECT_NOT_FOUND
            ], 200);
        }
        foreach ($data['structure'] as $pageId => $sections) {
            foreach ($sections as $sectionId => $fields) {
                foreach ($fields as $fieldId => $content) {
                    SiteMap::where([
                        'project_id' => $project->id,
                        'page_id' => $pageId,
                        'section_id' => $sectionId,
                        'field_id' => $fieldId,
                    ])->update(['content' => $content]);
                }
            }
        }
        return $this->JsonAbort([
            'status' => 'success',
        ], 200);
    }

    public function updateSiteMapPages()
    {
        $this->retrieveUser();
        $user = $this->user;
        $project = Project::find($user->project->id);

        $data = $this->getValidatedInput(
            ['structure'],
            [
                'structure' => 'nullable|array',
                'structure.*.content' => 'nullable|string',
                'structure.*.image' => 'nullable|string',
            ]
        );

        if (!$project) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => ErrorCodes::PROJECT_NOT_FOUND
            ], 200);
        }

        foreach ($data['structure'] as $pageId => $fileData) {
            $sitemapPage = SiteMap::where('project_id', $project->id)
                ->where('page_id', $pageId)
                ->first();

            if (!$sitemapPage) {
                return response()->json([
                    'status' => 'error',
                    'code' => ErrorCodes::PAGE_NOT_FOUND,
                    'page_id' => $pageId
                ], 404);
            }
            // Handle content file if provided
            if (!empty($fileData['content'])) {
                $file = $fileData['content'];
                $contentFile = $this->getFileFromBase64($file);
                $sitemapPage->content_document = $contentFile;
                $sitemapPage->content = $contentFile->path;
            }

            // Handle image file if provided
            if (!empty($fileData['image'])) {
                $image = $fileData['image'];
                $imageFile = $this->getFileFromBase64($image);
                $sitemapPage->content_images = $imageFile;
            }
            $sitemapPage->save();
        }

        return $this->JsonAbort([
            'status' => 'success'
        ], 200);
    }


}
