<?php
namespace Majormedia\Structure\Updates;

use Seeder;
use Illuminate\Support\Facades\DB;
use MajorMedia\Structure\Models\Feature;

/**
 * CreateFeatureSeeder Seeder
 *
 * Seeds predefined features into the database.
 * Project Types: 1 = E-commerce (PrestaShop), 2 = Vitrine (WordPress), 3 = Landing Page
 */
class CreateFeatureSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run()
    {
        // ============================================
        // FEATURES DEFINITIONS
        // ============================================
        // Each feature can be linked to multiple project types
        // Structure: title, description, price, sort_order, project_types => [project_type_id => [is_primary, is_required]]

        $featuresData = [
            // ==========================================
            // FONCTIONNALITÉS COMMUNES (Tous les types)
            // ==========================================
            [
                'title' => 'Intégration Google Analytics / Google Tag Manager',
                'description' => 'Intégration complète de Google Analytics 4 et Google Tag Manager pour le suivi des performances, des conversions et du comportement des utilisateurs.',
                'price' => 150.00,
                'sort_order' => 1,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                    2 => ['is_primary' => 1, 'is_required' => 0],
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Intégration Facebook Pixel',
                'description' => 'Configuration du Facebook Pixel pour le suivi des conversions, le remarketing et la création d\'audiences personnalisées.',
                'price' => 100.00,
                'sort_order' => 2,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                    2 => ['is_primary' => 0, 'is_required' => 0],
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Cookies et conformité RGPD',
                'description' => 'Bannière de consentement aux cookies conforme au RGPD avec gestion des préférences utilisateur et documentation légale.',
                'price' => 200.00,
                'sort_order' => 3,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 1],
                    2 => ['is_primary' => 1, 'is_required' => 1],
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Intégration de Captcha (reCAPTCHA)',
                'description' => 'Protection des formulaires contre le spam avec Google reCAPTCHA v2 ou v3.',
                'price' => 80.00,
                'sort_order' => 4,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                    2 => ['is_primary' => 1, 'is_required' => 0],
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Certificat SSL et HTTPS',
                'description' => 'Installation et configuration du certificat SSL pour sécuriser les échanges de données.',
                'price' => 50.00,
                'sort_order' => 5,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 1],
                    2 => ['is_primary' => 1, 'is_required' => 1],
                    3 => ['is_primary' => 1, 'is_required' => 1],
                ]
            ],
            [
                'title' => 'Optimisation SEO de base',
                'description' => 'Configuration des balises meta, URLs optimisées, sitemap XML, robots.txt et structure de données Schema.org.',
                'price' => 300.00,
                'sort_order' => 6,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                    2 => ['is_primary' => 1, 'is_required' => 0],
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Optimisation des performances (vitesse)',
                'description' => 'Optimisation du temps de chargement : compression images, mise en cache, minification CSS/JS, lazy loading.',
                'price' => 250.00,
                'sort_order' => 7,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                    2 => ['is_primary' => 1, 'is_required' => 0],
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Design responsive (mobile-first)',
                'description' => 'Adaptation du design pour tous les appareils : mobile, tablette et desktop.',
                'price' => 0.00,
                'sort_order' => 8,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 1],
                    2 => ['is_primary' => 1, 'is_required' => 1],
                    3 => ['is_primary' => 1, 'is_required' => 1],
                ]
            ],
            [
                'title' => 'Intégration réseaux sociaux',
                'description' => 'Liens vers les réseaux sociaux, boutons de partage et intégration des feeds sociaux.',
                'price' => 100.00,
                'sort_order' => 9,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                    2 => ['is_primary' => 1, 'is_required' => 0],
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Formulaire de contact avancé',
                'description' => 'Formulaire de contact personnalisé avec validation, notifications email et anti-spam.',
                'price' => 120.00,
                'sort_order' => 10,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                    2 => ['is_primary' => 1, 'is_required' => 0],
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Chat en direct (WhatsApp / Messenger / Tawk.to)',
                'description' => 'Intégration d\'un widget de chat en direct pour le support client instantané.',
                'price' => 100.00,
                'sort_order' => 11,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                    2 => ['is_primary' => 0, 'is_required' => 0],
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Intégration Google Maps',
                'description' => 'Affichage d\'une carte interactive avec localisation, itinéraire et informations de contact.',
                'price' => 80.00,
                'sort_order' => 12,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                    2 => ['is_primary' => 1, 'is_required' => 0],
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Newsletter (Mailchimp / Sendinblue / MailerLite)',
                'description' => 'Intégration d\'un service d\'emailing avec formulaire d\'inscription et synchronisation automatique.',
                'price' => 150.00,
                'sort_order' => 13,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                    2 => ['is_primary' => 0, 'is_required' => 0],
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Sauvegarde automatique',
                'description' => 'Configuration de sauvegardes automatiques quotidiennes ou hebdomadaires avec restauration facile.',
                'price' => 100.00,
                'sort_order' => 14,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                    2 => ['is_primary' => 1, 'is_required' => 0],
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Maintenance et mises à jour (1 an)',
                'description' => 'Service de maintenance incluant mises à jour de sécurité, corrections de bugs et support technique.',
                'price' => 500.00,
                'sort_order' => 15,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                    2 => ['is_primary' => 0, 'is_required' => 0],
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - GESTION PRODUITS
            // ==========================================
            [
                'title' => 'Variations de produits (Combinaisons)',
                'description' => 'Gestion des déclinaisons produits : taille, couleur, matière avec prix et stock spécifiques.',
                'price' => 0.00,
                'sort_order' => 20,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 1],
                ]
            ],
            [
                'title' => 'Packs et lots de produits',
                'description' => 'Création de packs de produits avec prix réduit et gestion automatique des stocks.',
                'price' => 150.00,
                'sort_order' => 21,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Produits virtuels / téléchargeables',
                'description' => 'Vente de produits dématérialisés : ebooks, logiciels, fichiers avec livraison automatique.',
                'price' => 200.00,
                'sort_order' => 22,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Produits personnalisables',
                'description' => 'Champs de personnalisation sur les produits : texte, fichier, options avec surcoût.',
                'price' => 250.00,
                'sort_order' => 23,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Galerie d\'images produit avancée',
                'description' => 'Multiple images par produit avec zoom, 360°, vidéo produit et images par déclinaison.',
                'price' => 150.00,
                'sort_order' => 24,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Fiche produit enrichie (onglets)',
                'description' => 'Onglets personnalisés sur la fiche produit : description, caractéristiques, documents, vidéos.',
                'price' => 100.00,
                'sort_order' => 25,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Comparateur de produits',
                'description' => 'Fonctionnalité permettant aux clients de comparer les caractéristiques de plusieurs produits.',
                'price' => 200.00,
                'sort_order' => 26,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Guide des tailles',
                'description' => 'Tableau des tailles interactif avec guide de mesure pour les produits vestimentaires.',
                'price' => 100.00,
                'sort_order' => 27,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - GESTION CATALOGUE
            // ==========================================
            [
                'title' => 'Importation / Exportation CSV',
                'description' => 'Import/export en masse des produits, catégories, clients, commandes via fichiers CSV.',
                'price' => 150.00,
                'sort_order' => 30,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Importation d\'images en masse',
                'description' => 'Import d\'images depuis un dossier avec association automatique par référence ou nom.',
                'price' => 100.00,
                'sort_order' => 31,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Synchronisation API externe',
                'description' => 'Connexion à une API externe pour synchroniser produits, stocks et prix automatiquement.',
                'price' => 500.00,
                'sort_order' => 32,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Gestion multi-fournisseurs',
                'description' => 'Association de plusieurs fournisseurs par produit avec prix d\'achat et références.',
                'price' => 150.00,
                'sort_order' => 33,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Gestion des marques / fabricants',
                'description' => 'Pages dédiées aux marques avec logo, description et liste des produits associés.',
                'price' => 100.00,
                'sort_order' => 34,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Attributs et caractéristiques avancés',
                'description' => 'Création d\'attributs personnalisés avec filtres de recherche et affichage sur fiche produit.',
                'price' => 150.00,
                'sort_order' => 35,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Recherche avancée avec filtres (facettes)',
                'description' => 'Navigation à facettes permettant de filtrer par prix, marque, attributs, disponibilité.',
                'price' => 200.00,
                'sort_order' => 36,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Recherche prédictive (auto-complétion)',
                'description' => 'Suggestions de recherche en temps réel avec aperçu des produits correspondants.',
                'price' => 150.00,
                'sort_order' => 37,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - STOCK & COMMANDES
            // ==========================================
            [
                'title' => 'Gestion des stocks avancée',
                'description' => 'Suivi des stocks en temps réel, alertes rupture, gestion multi-entrepôts.',
                'price' => 200.00,
                'sort_order' => 40,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Alerte disponibilité produit',
                'description' => 'Notification email aux clients quand un produit en rupture redevient disponible.',
                'price' => 100.00,
                'sort_order' => 41,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Précommande',
                'description' => 'Permettre aux clients de commander des produits avant leur disponibilité.',
                'price' => 150.00,
                'sort_order' => 42,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Gestion des retours (RMA)',
                'description' => 'Module de gestion des retours produits avec suivi, motifs et remboursement.',
                'price' => 200.00,
                'sort_order' => 43,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Facturation automatique PDF',
                'description' => 'Génération automatique des factures et avoirs en PDF avec envoi par email.',
                'price' => 100.00,
                'sort_order' => 44,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Bon de livraison personnalisé',
                'description' => 'Génération de bons de livraison personnalisés avec informations de suivi.',
                'price' => 80.00,
                'sort_order' => 45,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Emails transactionnels personnalisés',
                'description' => 'Personnalisation des emails de confirmation, expédition, suivi avec template sur mesure.',
                'price' => 150.00,
                'sort_order' => 46,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - PRIX & PROMOTIONS
            // ==========================================
            [
                'title' => 'Règles de prix catalogue',
                'description' => 'Réductions automatiques basées sur catégorie, marque, quantité ou groupe client.',
                'price' => 150.00,
                'sort_order' => 50,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Codes promo et bons de réduction',
                'description' => 'Création de codes promo avec conditions : montant minimum, produits spécifiques, durée.',
                'price' => 100.00,
                'sort_order' => 51,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Prix dégressifs (quantité)',
                'description' => 'Tarification dégressive selon la quantité commandée.',
                'price' => 100.00,
                'sort_order' => 52,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Prix par groupe client (B2B)',
                'description' => 'Tarifs différenciés selon les groupes de clients : particuliers, professionnels, VIP.',
                'price' => 200.00,
                'sort_order' => 53,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Ventes flash avec compte à rebours',
                'description' => 'Promotions limitées dans le temps avec affichage d\'un compte à rebours.',
                'price' => 150.00,
                'sort_order' => 54,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Offres "X achetés, Y offert"',
                'description' => 'Promotions du type 2+1 gratuit, lot de 3 au prix de 2, etc.',
                'price' => 150.00,
                'sort_order' => 55,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Programme de parrainage',
                'description' => 'Système de parrainage avec récompenses pour le parrain et le filleul.',
                'price' => 250.00,
                'sort_order' => 56,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Programme de fidélité (points)',
                'description' => 'Système de points de fidélité convertibles en réductions sur les achats.',
                'price' => 300.00,
                'sort_order' => 57,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - PAIEMENT
            // ==========================================
            [
                'title' => 'Paiement par carte bancaire (Stripe)',
                'description' => 'Intégration Stripe pour paiement CB sécurisé avec 3D Secure.',
                'price' => 200.00,
                'sort_order' => 60,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Paiement PayPal',
                'description' => 'Intégration PayPal Checkout avec paiement express et paiement en 4x.',
                'price' => 150.00,
                'sort_order' => 61,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Paiement à la livraison (COD)',
                'description' => 'Option de paiement en espèces à la réception de la commande.',
                'price' => 50.00,
                'sort_order' => 62,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Paiement par virement bancaire',
                'description' => 'Option de paiement par virement avec instructions automatiques.',
                'price' => 50.00,
                'sort_order' => 63,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Paiement en plusieurs fois (Alma / Oney)',
                'description' => 'Paiement fractionné en 3x ou 4x sans frais pour le client.',
                'price' => 300.00,
                'sort_order' => 64,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Paiement CMI (Maroc)',
                'description' => 'Intégration du Centre Monétique Interbancaire pour les cartes marocaines.',
                'price' => 300.00,
                'sort_order' => 65,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Paiement mobile (Apple Pay / Google Pay)',
                'description' => 'Paiement express via portefeuilles mobiles Apple Pay et Google Pay.',
                'price' => 200.00,
                'sort_order' => 66,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - LIVRAISON
            // ==========================================
            [
                'title' => 'Transporteurs multiples',
                'description' => 'Configuration de plusieurs transporteurs avec tarifs par zone, poids et prix.',
                'price' => 150.00,
                'sort_order' => 70,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Livraison gratuite conditionnelle',
                'description' => 'Livraison offerte à partir d\'un certain montant ou pour certains produits.',
                'price' => 50.00,
                'sort_order' => 71,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Points relais (Mondial Relay / Colissimo)',
                'description' => 'Intégration de la livraison en point relais avec carte interactive.',
                'price' => 200.00,
                'sort_order' => 72,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Suivi de colis en temps réel',
                'description' => 'Tracking automatique avec numéro de suivi et notifications de livraison.',
                'price' => 150.00,
                'sort_order' => 73,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Calcul des frais par ville/zone',
                'description' => 'Tarification personnalisée par ville, région ou zone géographique.',
                'price' => 150.00,
                'sort_order' => 74,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Click & Collect (retrait en magasin)',
                'description' => 'Option de retrait en boutique avec choix de la date et du créneau horaire.',
                'price' => 200.00,
                'sort_order' => 75,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Livraison express / Same day',
                'description' => 'Option de livraison le jour même ou en express avec surcoût.',
                'price' => 150.00,
                'sort_order' => 76,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - CLIENTS
            // ==========================================
            [
                'title' => 'Espace client complet',
                'description' => 'Tableau de bord client : commandes, adresses, favoris, informations personnelles.',
                'price' => 0.00,
                'sort_order' => 80,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 1],
                ]
            ],
            [
                'title' => 'Liste de souhaits (Wishlist)',
                'description' => 'Permettre aux clients de sauvegarder des produits pour plus tard.',
                'price' => 100.00,
                'sort_order' => 81,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Connexion sociale (Google / Facebook)',
                'description' => 'Inscription et connexion via les comptes Google et Facebook.',
                'price' => 150.00,
                'sort_order' => 82,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Commande invité (sans inscription)',
                'description' => 'Permettre l\'achat sans création de compte obligatoire.',
                'price' => 0.00,
                'sort_order' => 83,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Compte professionnel B2B',
                'description' => 'Espace dédié aux professionnels avec validation, prix HT et documents.',
                'price' => 300.00,
                'sort_order' => 84,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Groupes de clients personnalisés',
                'description' => 'Création de groupes avec accès et tarifs spécifiques.',
                'price' => 150.00,
                'sort_order' => 85,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - MARKETING
            // ==========================================
            [
                'title' => 'Avis et notes clients',
                'description' => 'Système d\'avis produits avec modération, notes et photos clients.',
                'price' => 150.00,
                'sort_order' => 90,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Produits vus récemment',
                'description' => 'Affichage des derniers produits consultés par le visiteur.',
                'price' => 50.00,
                'sort_order' => 91,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Cross-selling (produits associés)',
                'description' => 'Suggestions de produits complémentaires sur la fiche produit et au panier.',
                'price' => 100.00,
                'sort_order' => 92,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Up-selling (montée en gamme)',
                'description' => 'Suggestions de produits similaires de gamme supérieure.',
                'price' => 100.00,
                'sort_order' => 93,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Email panier abandonné',
                'description' => 'Envoi automatique d\'emails de relance pour les paniers non finalisés.',
                'price' => 200.00,
                'sort_order' => 94,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Pop-up promotionnel',
                'description' => 'Pop-up d\'entrée ou de sortie avec offre spéciale ou collecte d\'emails.',
                'price' => 100.00,
                'sort_order' => 95,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Bannières promotionnelles dynamiques',
                'description' => 'Gestion de bannières avec planification et ciblage par catégorie.',
                'price' => 100.00,
                'sort_order' => 96,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - MULTI & INTERNATIONAL
            // ==========================================
            [
                'title' => 'Multi-langues',
                'description' => 'Support de plusieurs langues avec traduction du catalogue et de l\'interface.',
                'price' => 200.00,
                'sort_order' => 100,
                'project_types' => [
                    1 => ['is_primary' => 1, 'is_required' => 0],
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Multi-devises',
                'description' => 'Affichage des prix dans plusieurs devises avec conversion automatique.',
                'price' => 150.00,
                'sort_order' => 101,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Multi-boutiques',
                'description' => 'Gestion de plusieurs boutiques depuis une seule installation PrestaShop.',
                'price' => 500.00,
                'sort_order' => 102,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Taxes et TVA par pays',
                'description' => 'Configuration des taxes selon le pays de livraison (intra-communautaire, export).',
                'price' => 150.00,
                'sort_order' => 103,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // E-COMMERCE PRESTASHOP - INTÉGRATIONS
            // ==========================================
            [
                'title' => 'Intégration ERP / Logiciel de gestion',
                'description' => 'Connexion avec votre ERP pour synchroniser stocks, commandes et clients.',
                'price' => 800.00,
                'sort_order' => 110,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Intégration CRM (HubSpot / Salesforce)',
                'description' => 'Synchronisation des contacts et commandes avec votre CRM.',
                'price' => 400.00,
                'sort_order' => 111,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Export vers marketplaces (Amazon / Cdiscount)',
                'description' => 'Publication automatique des produits sur les principales marketplaces.',
                'price' => 500.00,
                'sort_order' => 112,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Google Shopping / Merchant Center',
                'description' => 'Flux produits pour Google Shopping avec synchronisation automatique.',
                'price' => 250.00,
                'sort_order' => 113,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Instagram Shopping / Facebook Shop',
                'description' => 'Synchronisation du catalogue pour la vente sur les réseaux sociaux.',
                'price' => 250.00,
                'sort_order' => 114,
                'project_types' => [
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // VITRINE WORDPRESS - FONCTIONNALITÉS DE BASE
            // ==========================================
            [
                'title' => 'Thème WordPress personnalisé',
                'description' => 'Création d\'un thème sur mesure ou personnalisation avancée d\'un thème premium.',
                'price' => 0.00,
                'sort_order' => 120,
                'project_types' => [
                    2 => ['is_primary' => 1, 'is_required' => 1],
                ]
            ],
            [
                'title' => 'Page builder (Elementor Pro)',
                'description' => 'Installation et configuration d\'Elementor Pro pour édition visuelle facile.',
                'price' => 150.00,
                'sort_order' => 121,
                'project_types' => [
                    2 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Blog intégré',
                'description' => 'Configuration complète du blog : catégories, tags, commentaires, partage social.',
                'price' => 100.00,
                'sort_order' => 122,
                'project_types' => [
                    2 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Portfolio / Galerie de projets',
                'description' => 'Section portfolio avec filtres, lightbox et pages de détail des projets.',
                'price' => 200.00,
                'sort_order' => 123,
                'project_types' => [
                    2 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Page équipe',
                'description' => 'Présentation des membres de l\'équipe avec photos, postes et liens sociaux.',
                'price' => 100.00,
                'sort_order' => 124,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Section témoignages / avis clients',
                'description' => 'Carrousel ou grille de témoignages clients avec photos et notes.',
                'price' => 100.00,
                'sort_order' => 125,
                'project_types' => [
                    2 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'FAQ accordéon',
                'description' => 'Section questions fréquentes avec système d\'accordéon interactif.',
                'price' => 80.00,
                'sort_order' => 126,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Timeline / Historique',
                'description' => 'Présentation chronologique de l\'histoire de l\'entreprise.',
                'price' => 100.00,
                'sort_order' => 127,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Compteurs animés (chiffres clés)',
                'description' => 'Animation de compteurs pour mettre en avant les statistiques de l\'entreprise.',
                'price' => 50.00,
                'sort_order' => 128,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // VITRINE WORDPRESS - MÉDIAS & CONTENU
            // ==========================================
            [
                'title' => 'Galerie photos avancée',
                'description' => 'Galerie d\'images avec filtres, lightbox, zoom et diaporama.',
                'price' => 100.00,
                'sort_order' => 130,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Intégration vidéo (YouTube / Vimeo)',
                'description' => 'Intégration de vidéos avec lecture automatique, lazy loading et responsive.',
                'price' => 50.00,
                'sort_order' => 131,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Slider / Carrousel héro',
                'description' => 'Carrousel d\'images ou de contenus en page d\'accueil avec animations.',
                'price' => 100.00,
                'sort_order' => 132,
                'project_types' => [
                    2 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Animations au scroll',
                'description' => 'Effets d\'animation des éléments lors du défilement de la page.',
                'price' => 100.00,
                'sort_order' => 133,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Parallax effet',
                'description' => 'Effet de parallaxe sur les images et sections pour un design moderne.',
                'price' => 80.00,
                'sort_order' => 134,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Téléchargement de fichiers (PDF, docs)',
                'description' => 'Zone de téléchargement pour brochures, catalogues et documents.',
                'price' => 80.00,
                'sort_order' => 135,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // VITRINE WORDPRESS - FONCTIONNALITÉS AVANCÉES
            // ==========================================
            [
                'title' => 'Système de réservation / prise de RDV',
                'description' => 'Calendrier de réservation en ligne avec notifications et rappels.',
                'price' => 350.00,
                'sort_order' => 140,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Devis en ligne',
                'description' => 'Formulaire de demande de devis personnalisé avec calcul automatique.',
                'price' => 250.00,
                'sort_order' => 141,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Espace membres / client',
                'description' => 'Zone privée accessible sur inscription avec contenu exclusif.',
                'price' => 300.00,
                'sort_order' => 142,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Forum de discussion',
                'description' => 'Espace communautaire avec forums de discussion et profils utilisateurs.',
                'price' => 400.00,
                'sort_order' => 143,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Module d\'offres d\'emploi',
                'description' => 'Section carrières avec liste des postes et formulaire de candidature.',
                'price' => 200.00,
                'sort_order' => 144,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Agenda / Calendrier d\'événements',
                'description' => 'Liste des événements avec calendrier, inscriptions et rappels.',
                'price' => 250.00,
                'sort_order' => 145,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Multi-sites WordPress',
                'description' => 'Gestion de plusieurs sites depuis une seule installation WordPress.',
                'price' => 400.00,
                'sort_order' => 146,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Intranet / Extranet',
                'description' => 'Espace privé pour employés ou partenaires avec documents et outils.',
                'price' => 500.00,
                'sort_order' => 147,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // VITRINE WORDPRESS - SÉCURITÉ & TECHNIQUE
            // ==========================================
            [
                'title' => 'Sécurité renforcée WordPress',
                'description' => 'Protection contre les attaques : firewall, limite de connexions, scan malware.',
                'price' => 150.00,
                'sort_order' => 150,
                'project_types' => [
                    2 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'CDN (Content Delivery Network)',
                'description' => 'Configuration d\'un CDN pour améliorer la vitesse de chargement mondiale.',
                'price' => 100.00,
                'sort_order' => 151,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Cache avancé (Redis / Varnish)',
                'description' => 'Mise en cache serveur pour des performances optimales.',
                'price' => 150.00,
                'sort_order' => 152,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Migration de site existant',
                'description' => 'Migration complète d\'un site existant vers la nouvelle plateforme.',
                'price' => 300.00,
                'sort_order' => 153,
                'project_types' => [
                    2 => ['is_primary' => 0, 'is_required' => 0],
                    1 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],

            // ==========================================
            // LANDING PAGE - FONCTIONNALITÉS SPÉCIFIQUES
            // ==========================================
            [
                'title' => 'Design one-page optimisé conversion',
                'description' => 'Page unique avec sections stratégiques pour maximiser les conversions.',
                'price' => 0.00,
                'sort_order' => 160,
                'project_types' => [
                    3 => ['is_primary' => 1, 'is_required' => 1],
                ]
            ],
            [
                'title' => 'Formulaire de capture de leads',
                'description' => 'Formulaire optimisé pour la collecte de prospects avec intégration CRM/email.',
                'price' => 100.00,
                'sort_order' => 161,
                'project_types' => [
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'A/B Testing',
                'description' => 'Configuration de tests A/B pour optimiser les taux de conversion.',
                'price' => 200.00,
                'sort_order' => 162,
                'project_types' => [
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Compte à rebours (urgence)',
                'description' => 'Timer dynamique créant un sentiment d\'urgence pour l\'offre.',
                'price' => 80.00,
                'sort_order' => 163,
                'project_types' => [
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Pop-up d\'intention de sortie',
                'description' => 'Pop-up s\'affichant quand le visiteur s\'apprête à quitter la page.',
                'price' => 100.00,
                'sort_order' => 164,
                'project_types' => [
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Barre de progression',
                'description' => 'Indicateur visuel de progression dans un processus multi-étapes.',
                'price' => 50.00,
                'sort_order' => 165,
                'project_types' => [
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Preuve sociale dynamique',
                'description' => 'Notifications en temps réel des inscriptions/achats récents.',
                'price' => 150.00,
                'sort_order' => 166,
                'project_types' => [
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Vidéo de vente intégrée',
                'description' => 'Lecteur vidéo optimisé avec auto-play et contrôles personnalisés.',
                'price' => 100.00,
                'sort_order' => 167,
                'project_types' => [
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Section tarification',
                'description' => 'Tableau de prix comparatif avec mise en avant de l\'offre recommandée.',
                'price' => 100.00,
                'sort_order' => 168,
                'project_types' => [
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Bouton CTA flottant',
                'description' => 'Bouton d\'action toujours visible lors du défilement de la page.',
                'price' => 50.00,
                'sort_order' => 169,
                'project_types' => [
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Page de remerciement personnalisée',
                'description' => 'Page après conversion avec offre additionnelle ou instructions.',
                'price' => 100.00,
                'sort_order' => 170,
                'project_types' => [
                    3 => ['is_primary' => 1, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Intégration Calendly / Cal.com',
                'description' => 'Prise de rendez-vous directement sur la landing page.',
                'price' => 100.00,
                'sort_order' => 171,
                'project_types' => [
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Pixels de tracking avancés (TikTok, LinkedIn)',
                'description' => 'Intégration des pixels publicitaires pour remarketing multi-plateforme.',
                'price' => 150.00,
                'sort_order' => 172,
                'project_types' => [
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
            [
                'title' => 'Intégration paiement (Stripe Checkout)',
                'description' => 'Paiement direct sur la landing page pour vente de produits/services.',
                'price' => 200.00,
                'sort_order' => 173,
                'project_types' => [
                    3 => ['is_primary' => 0, 'is_required' => 0],
                ]
            ],
        ];

        // ============================================
        // INSERT FEATURES INTO DATABASE
        // ============================================

        foreach ($featuresData as $featureData) {
            // Extract project_types before inserting
            $projectTypes = $featureData['project_types'];
            unset($featureData['project_types']);

            // Insert feature
            $featureData['project_id'] = null;
            DB::table('majormedia_structure_features')->insert($featureData);

            // Get the inserted feature ID
            $featureId = Feature::where('title', $featureData['title'])->value('id');

            // Link to project types
            foreach ($projectTypes as $projectTypeId => $settings) {
                DB::table('majormedia_structure_project_types_features')->insert([
                    'project_type_id' => $projectTypeId,
                    'feature_id' => $featureId,
                    'is_primary' => $settings['is_primary'],
                    'is_required' => $settings['is_required'],
                ]);
            }
        }
    }
}
