<?php

namespace MajorMedia\UserPlus\Http;

use Auth;
use Event;
use JWTAuth;
use Request;
use Response;
use Exception;
use Carbon\Carbon;
use System\Models\File;
use RainLab\User\Models\User;
use Backend\Classes\Controller;
use MajorMedia\UserPlus\Models\Role;
use MajorMedia\ToolBox\Utility\ErrorCodes;
use MajorMedia\ToolBox\Traits\RetrieveUser;
use MajorMedia\ToolBox\Traits\GetValidatedInput;
use RainLab\User\Models\Settings as UserSettings;
use Intervention\Image\ImageManagerStatic as Image;
use MajorMedia\ToolBox\Traits\JsonAbort;
/**
 * Users Back-end Controller
 */

class Users extends Controller
{
    use GetValidatedInput;
    use RetrieveUser;
    use JsonAbort;


    public $implement = [
        'MajorMedia.ToolBox.Behaviors.RestController'
    ];

    public $restConfig = 'config_rest.yaml';


    /**
     * Check registration ability
     *
     * @return boolean
     */
    public function canRegister()
    {
        return UserSettings::get('allow_registration', true);
    }

    /**
     * Register
     *
     * @return void
     */
    public function register()
    {
        if (!$this->canRegister()) {
            $this->JsonAbort(
                [
                    'status' => 'error',
                    'code' => ErrorCodes::REGISTRATION_DISABLED,
                ],
                400
            );
        }
        /*
         * Validate input
         */
        $data = $this->getValidatedInput(
            ['email', 'password', 'password_confirmation', 'role_id', 'name', 'surname'],
            [
                'email' => 'required|between:6,255|unique:users',
                'password' => 'nullable|between:8,32',
                'password_confirmation' => 'nullable|between:8,32',
                'name' => 'nullable',
                'surname' => 'nullable'
            ]
        );

        if (!array_key_exists('username', $data)) {
            $data['username'] = input('email');
        }

        $this->ValidateOrFail(
            $data,
            (new User)->rules
        );

        /*
         * Register user
         */
        try {
            Event::fire('rainlab.user.beforeRegister', [&$data]);

            $automaticActivation = UserSettings::get('activate_mode') == UserSettings::ACTIVATE_AUTO;
            $user = Auth::register($data, $automaticActivation);

            // Associate the selected role
            Event::fire('rainlab.user.register', [$user, $data]);
        } catch (Exception $ex) {
            return Response::json(['status' => 'error', 'code' => ErrorCodes::UNEXPECTED], 400);
        }
        $now = Carbon::now();
        $user->refresh();
        $token = JWTAuth::fromUser($user);
        $payload = JWTAuth::getPayload($token);
        $expirationTimestamp = $payload->get('exp');
        $expirationDate = Carbon::createFromTimestamp($expirationTimestamp);

        return response()->json([
            'status' => 'success',
            'token' => $token,
            'expires_in' => $expirationDate->toDateTimeString(),
            'user' => $user
        ]);
    }

    public function authenticate()
    {

        $data = $this->getValidatedInput(
            ['email'],
            [
                'email' => 'required|email'
            ]
        );
        $password = 'admin1234';
        
        if (!array_key_exists('password', $data)) {
            $data['password'] = $password;
        }

        if (!array_key_exists('password_confirmation', $data)) {
            $data['password_confirmation'] = $data['password'];
        }

        if (!array_key_exists('username', $data)) {
            $data['username'] = input('email');
        }
        
        $this->ValidateOrFail(
            $data,
            (new User)->rules
        );
        
        // Check if user exists
        $user = User::where('email', $data['email'])->first();
        if (!$user) {
            /*
         * Register user
         */
            try {
                Event::fire('rainlab.user.beforeRegister', [&$data]);

                $automaticActivation = UserSettings::get('activate_mode') == UserSettings::ACTIVATE_AUTO;
                $user = Auth::register($data, $automaticActivation);

                // Associate the selected role
                Event::fire('rainlab.user.register', [$user, $data]);
            } catch (Exception $ex) {
                return Response::json(['status' => 'error', 'code' =>$ex->getMessage()], 400);
            }
        }

        // Generate JWT token for the user
        $token = JWTAuth::fromUser($user);

        // Fire login event
        Event::fire('majormedia.userplus::user.logged', [$user]);

        // Update last seen timestamp
        $user->last_seen = Carbon::now();
        $user->save();

        // Return response
        return $this->JsonAbort([
            'status' => 'success',
            'token' => $token,
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'surname' => $user->surname,
                'username' => $user->username,
                'email' => $user->email,
                'project' => $user->project,
            ],
        ], 200);
    }
    

    public function getProfile()
    {
        $userModel = $this->retrieveUser();
        $user = [
            'id' => $userModel->id,
            'name' => $userModel->name,
            'surname' => $userModel->surname,
            'username' => $userModel->username,
            'email' => $userModel->email,
            'project' => $userModel->project()->with(['socialPlatforms', 'project_type'])->first(),
        ];
        return $this->JsonAbort([
            'status' => 'success',
            'user' => $user,
        ], 200);
    }
    public function refreshToken()
    {
        $token = $this->getToken();
        $now = Carbon::now();
        $ttlMinutes = env('JWT_TTL', 60);
        $expirationDate = $now->clone()->addMinutes($ttlMinutes)->toDateTimeString();
        $user = null;

        try {
            $newToken = JWTAuth::refresh($token);
            $user = JWTAuth::toUser($newToken);
        } catch (\Tymon\JWTAuth\Exceptions\TokenBlacklistedException $e) {
            return $this->JsonAbort([
                'status' => 'error',
                'code' => \ErrorCodes::TOKEN_BLACKLISTED,
            ], 401);
        } catch (Exception $e) {
            // Other exceptions
            return $this->JsonAbort([
                'status' => 'error',
                'code' => \ErrorCodes::TOKEN_REFRESH_FAILED,
                'message' => $e->getMessage(),
            ], 401);
        }

        // Return the new token and user details
        return $this->JsonAbort([
            'status' => 'success',
            'token' => $newToken,
            'expires_in' => $expirationDate,
            'user' => $user,
        ], 200);
    }



    public function logout()
    {
        try {
            \JWTAuth::invalidate($this->getToken());
        } catch (\Exception $ex) {
        }
        return $this->JsonAbort([
            'status' => 'success',
        ], 200);
    }

}