<?php namespace October\Rain\Parse;

use Html;
use Event;
use October\Rain\Parse\Parsedown\ParsedownExtra;

/**
 * Markdown helper class.
 *
 * Calling Markdown::parse($text) returns the HTML corresponding
 * to the Markdown input in $text.
 *
 * October CMS uses ParsedownExtra as its Markdown parser,
 * but fires markdown.beforeParse and markdown.parse events
 * allowing hooks into the default parsing,
 *
 * The markdown.beforeParse event passes a MarkdownData
 * instance, containing a public $text variable. Event
 * listeners can modify $text, for example to filter out
 * or protect snippets from being interpreted by ParseDown.
 *
 * Similarly, markdown.parse is fired after ParseDown has
 * interpreted the (possibly modified) input. This event
 * passes an array [$text, $data], where $text is the
 * original unmodified Markdown input, and $data is the HTML
 * code generated by ParseDown.
 *
 * @package october\parse
 * @author Alexey Bobkov, Samuel Georges
 **/
class Markdown
{
    use \October\Rain\Support\Traits\Emitter;

    /**
     * @var ParsedownExtra parser is the parsedown instance
     */
    protected $parser;

    /**
     * parse text using Markdown and Markdown-Extra
     */
    public function parse($text): string
    {
        return $this->parseInternal($text);
    }

    /**
     * parseClean enables safe mode where the resulting HTML is cleaned
     * using a sanitizer
     */
    public function parseClean($text): string
    {
        $result = Html::clean($this->parse($text));

        $this->parser = null;

        return $result;
    }

    /**
     * parseSafe is stricter than parse clean allowing no HTML at all
     * except for basic protocols such as https://, ftps://, mailto:, etc.
     */
    public function parseSafe($text): string
    {
        $this->getParser()->setSafeMode(true);

        $result = $this->parse($text);

        $this->parser = null;

        return $result;
    }

    /**
     * parseIndent disables code blocks caused by indentation
     */
    public function parseIndent($text): string
    {
        $this->getParser()->setIndentMode(false);

        $result = $this->parse($text);

        $this->parser = null;

        return $result;
    }

    /**
     * parseLine parses a single line
     */
    public function parseLine($text): string
    {
        return $this->parseInternal($text, 'line');
    }

    /**
     * parseInternal is an internal method for parsing
     */
    protected function parseInternal($text, $method = 'text'): string
    {
        $data = new MarkdownData($text);

        $this->fireEvent('beforeParse', $data, false);
        Event::fire('markdown.beforeParse', $data, false);

        $result = $data->text;

        $result = $this->getParser()->$method($result);

        $data->text = $result;

        // The markdown.parse gets passed both the original
        // input and the result so far.
        $this->fireEvent('parse', [$text, $data], false);
        Event::fire('markdown.parse', [$text, $data], false);

        return $data->text;
    }

    /**
     * getParser returns an instance of the parser
     */
    protected function getParser(): ParsedownExtra
    {
        if ($this->parser === null) {
            $this->parser = new ParsedownExtra;
        }

        return $this->parser;
    }
}
